import { getConnection } from '../db/connection.js';

export const insertReports = async (req, res) => {
  let connection;
  const { table_prefix, company_id, reports, report_type } = req.body;

  if (!table_prefix || !company_id || !Array.isArray(reports) || reports.length === 0) {
    return res.status(400).json({ error: 'Debe enviar table_prefix, company_id y un arreglo de reports válidos' });
  }

  if (!["region", "country"].includes(report_type)) {
    return res.status(400).json({ error: "report_type debe ser 'region' o 'country'" });
  }

  try {
    connection = await getConnection();
    if (report_type === "region") {
      const regionValues = reports.map(r => [
        r.year,
        r.month,
        r.category_id,
        r.subcategory_id,
        company_id,
        r.region || 0,
        r.produccion || 0,
        r.ventasPesos || 0,
        r.ventasVolumen || 0
      ]);

      await connection.query(
        `INSERT INTO ${table_prefix}_region_reports 
         (year, month, category_id, subcategory_id, company_id, region, produccion, ventas_pesos, ventas_volumen)
         VALUES ?`,
        [regionValues]
      );
    }

    // 🔹 Inserción para reportes nacionales
    if (report_type === "country") {
      const countryValues = reports.map(r => [
        r.year,
        r.month,
        r.category_id,
        r.subcategory_id,
        company_id,
        r.produccion || 0
      ]);

      await connection.query(
        `INSERT INTO ${table_prefix}_country_reports 
         (year, month, category_id, subcategory_id, company_id, produccion)
         VALUES ?`,
        [countryValues]
      );

      if (reports.length > 0) {
        const r = reports[0];
        await connection.query(
          `INSERT INTO ${table_prefix}_report_history (month, year, company_nit, edit_permission)
     VALUES (?, ?, ?, 0)`,
          [r.month, r.year, company_id]
        );
      }
    }

    // Respuesta final
    res.json({
      message: 'Reports insertados correctamente',
      inserted: {
        region: report_type === "region" ? reports.length : 0,
        country: report_type === "country" ? reports.length : 0,
        history: reports.length
      }
    });

  } catch (error) {
    console.error('Error insertando reports:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  } finally {
    if (connection) connection.release();
  }
};

export const getSpecificReport = async (req, res) => {
  let connection;
  const { table_prefix, company_nit, year, month } = req.body;

  if (!table_prefix || !company_nit) {
    console.warn("Falta table_prefix o company_nit");
    return res.status(400).json({ error: "Debes enviar table_prefix y company_nit" });
  }

  try {
    connection = await getConnection();

    const regionTable = `${table_prefix}_region_reports`;
    const countryTable = `${table_prefix}_country_reports`;

    // Construir condiciones dinámicas
    let filters = "WHERE company_id = ?";
    const params = [company_nit];

    if (year) {
      filters += " AND year = ?";
      params.push(year);
    }

    if (month) {
      filters += " AND month = ?";
      params.push(month);
    }

    // Consultar reportes regionales
    const [regionReports] = await connection.query(
      `
      SELECT 
        'region' AS report_type,
        id,
        year,
        month,
        category_id,
        subcategory_id,
        company_id,
        region,
        produccion,
        ventas_pesos,
        ventas_volumen
      FROM \`${regionTable}\`
      ${filters}
      `,
      params
    );

    // Consultar reportes nacionales
    const [countryReports] = await connection.query(
      `
      SELECT 
        'country' AS report_type,
        id,
        year,
        month,
        category_id,
        subcategory_id,
        company_id,
        NULL AS region,
        produccion,
        NULL AS ventas_pesos,
        NULL AS ventas_volumen
      FROM \`${countryTable}\`
      ${filters}
      `,
      params
    );

    // Combinar ambos
    const allReports = [...regionReports, ...countryReports];

    res.status(200).json({
      message: 'Reportes obtenidos correctamente',
      total: allReports.length,
      region_reports: regionReports.length,
      country_reports: countryReports.length,
      data: allReports
    });

  } catch (error) {
    console.error("❌ Error obteniendo reports:", error);
    res.status(500).json({ error: "Error obteniendo reports" });
  } finally {
    if (connection) connection.release();
  }
};


