import { getConnection } from '../db/connection.js';

export const getReportHistory = async (req, res) => {
  let connection;
  const { table_prefix, company_nit } = req.body;

  //  Validar entrada
  if (!table_prefix || !company_nit) {
    return res.status(400).json({ error: 'Debe enviar table_prefix y company_nit' });
  }

  try {
    connection = await getConnection();

    //Obtener histórico
    const [history] = await connection.query(
      `SELECT id, year, month, company_nit
       FROM ${table_prefix}_report_history
       WHERE company_nit = ?
       ORDER BY year DESC, month DESC`,
      [company_nit]
    );

    //Respuesta
    res.json({
      message: 'Histórico obtenido correctamente',
      count: history.length,
      history,
    });

  } catch (error) {
    console.error('Error obteniendo histórico:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  } finally {
    if (connection) connection.release();
  }
};

export const getEditableReports = async (req, res) => {
  let connection;
  const { table_prefix, company_nit } = req.body;

  //  Validar entrada
  if (!table_prefix || !company_nit) {
    return res.status(400).json({ error: 'Debe enviar table_prefix y company_nit' });
  }

  try {
    connection = await getConnection();

    //Obtener solo registros con edit_permission = 1
    const [editableReports] = await connection.query(
      `SELECT id, year, month, company_nit, edit_permission
       FROM ${table_prefix}_report_history
       WHERE company_nit = ? AND edit_permission = 1
       ORDER BY year DESC, month DESC`,
      [company_nit]
    );

    //Respuesta
    res.json({
      message: 'Reportes con permiso de edición obtenidos correctamente',
      count: editableReports.length,
      reports: editableReports,
    });

  } catch (error) {
    console.error('Error obteniendo reportes editables:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  } finally {
    if (connection) connection.release();
  }
};

export const deleteReportHistory = async (req, res) => {
  let connection;
  const { table_prefix, company_nit, year, month } = req.body;

  // 🔹 Validar entrada
  if (!table_prefix || !company_nit || !year || !month) {
    return res.status(400).json({
      error: 'Debe enviar table_prefix, company_nit, year y month',
    });
  }

  try {
    connection = await getConnection();
    await connection.beginTransaction();

    // 1. Eliminar reportes regionales
    const [regionResult] = await connection.query(
      `DELETE FROM ${table_prefix}_region_reports
       WHERE company_id = ? AND year = ? AND month = ?`,
      [company_nit, year, month]
    );

    // 2. Eliminar reportes nacionales
    const [countryResult] = await connection.query(
      `DELETE FROM ${table_prefix}_country_reports
       WHERE company_id = ? AND year = ? AND month = ?`,
      [company_nit, year, month]
    );

    // 3. Eliminar registro del histórico
    const [historyResult] = await connection.query(
      `DELETE FROM ${table_prefix}_report_history
       WHERE company_nit = ? AND year = ? AND month = ?`,
      [company_nit, year, month]
    );

    // Confirmar si se eliminó algo
    const totalDeleted =
      regionResult.affectedRows + countryResult.affectedRows + historyResult.affectedRows;

    if (totalDeleted === 0) {
      await connection.rollback();
      return res.status(404).json({
        message: 'No se encontraron registros para eliminar con esos datos',
      });
    }

    await connection.commit();

    res.json({
      message: 'Registros eliminados correctamente',
      deleted: {
        region: regionResult.affectedRows,
        country: countryResult.affectedRows,
        history: historyResult.affectedRows,
      },
    });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Error eliminando registros:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  } finally {
    if (connection) connection.release();
  }
};

export const updateEditPermission = async (req, res) => {
  let connection;
  const { table_prefix, company_nit, year, month } = req.body;

  //  Validar entrada
  if (!table_prefix || !company_nit || !year || !month) {
    return res.status(400).json({
      error: 'Debe enviar table_prefix, company_nit, year y month'
    });
  }

  try {
    connection = await getConnection();

    // 🔹 Actualizar edit_permission de 0 → 1
    const [result] = await connection.query(
      `UPDATE ${table_prefix}_report_history
       SET edit_permission = 1
       WHERE company_nit = ? AND year = ? AND month = ?`,
      [company_nit, year, month]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({
        message: 'No se encontró ningún registro con esos datos o ya estaba en 1'
      });
    }

    res.json({
      message: 'Permiso de edición actualizado correctamente',
      updated: result.affectedRows,
    });

  } catch (error) {
    console.error('Error actualizando permiso de edición:', error);
    res.status(500).json({ error: 'Error interno del servidor' });
  } finally {
    if (connection) connection.release();
  }
};
