import { getConnection } from "../db/connection.js";

export const saveColors = async (req, res) => {
    let connection;
    const { table_prefix, color_primary, color_secondary, color_tertiary, color_quaternary } = req.body;

    if (!table_prefix) {
        return res.status(400).json({ error: "Debes enviar table_prefix" });
    }

    if (!color_primary || !color_secondary || !color_tertiary || !color_quaternary) {
        return res.status(400).json({ error: "Debes enviar los 4 colores" });
    }

    const hexRegex = /^#([0-9A-Fa-f]{6})$/;
    if (![color_primary, color_secondary, color_tertiary, color_quaternary].every(c => hexRegex.test(c))) {
        return res.status(400).json({ error: "Formato inválido para uno o más colores" });
    }

    try {
        connection = await getConnection();
        const tableName = `${table_prefix}_configurations`;

        // Nos aseguramos que exista la tabla
        await connection.query(`
            CREATE TABLE IF NOT EXISTS \`${tableName}\` (
                id INT NOT NULL AUTO_INCREMENT PRIMARY KEY,
                config VARCHAR(255) NOT NULL UNIQUE,
                value VARCHAR(255) NOT NULL
            )
        `);

        const configs = [
            ["color_primary", color_primary],
            ["color_secondary", color_secondary],
            ["color_tertiary", color_tertiary],
            ["color_quaternary", color_quaternary]
        ];

        for (const [config, value] of configs) {
            // ¿Ya existe este config?
            const [rows] = await connection.query(
                `SELECT id FROM \`${tableName}\` WHERE config = ?`,
                [config]
            );

            if (rows.length > 0) {
                // Actualizar
                await connection.query(
                    `UPDATE \`${tableName}\` SET value = ? WHERE config = ?`,
                    [value, config]
                );
            } else {
                // Insertar nuevo
                await connection.query(
                    `INSERT INTO \`${tableName}\` (config, value) VALUES (?, ?)`,
                    [config, value]
                );
            }
        }

        res.json({
            msg: "Colores guardados correctamente",
            data: { color_primary, color_secondary, color_tertiary, color_quaternary }
        });

    } catch (error) {
        console.error("Error guardando colores:", error);
        res.status(500).json({ error: "Error guardando colores", details: error.message });
    } finally {
        if (connection) connection.release();
    }
};
export const getConfigurations = async (req, res) => {
    let connection;
    const { table_prefix } = req.body;

    if (!table_prefix) {
        return res.status(400).json({ error: "Debes enviar table_prefix" });
    }

    try {
        connection = await getConnection();
        const tableConfigs = `${table_prefix}_configurations`;
        const tableCategories = `${table_prefix}_categories`;
        const tableSubcategories = `${table_prefix}_subcategories`;

        // Traer configuraciones
        const [rowsConfigs] = await connection.query(
            `SELECT * FROM \`${tableConfigs}\``
        );

        const configObject = rowsConfigs.reduce((acc, row) => {
            acc[row.config] = row.value;
            return acc;
        }, {});

        // Traer categorías
        const [rowsCategories] = await connection.query(
            `SELECT * FROM \`${tableCategories}\``
        );

        // Traer subcategorías
        const [rowsSubcategories] = await connection.query(
            `SELECT * FROM \`${tableSubcategories}\``
        );

        res.json({
            msg: "Datos obtenidos correctamente",
            data: {
                configurations: configObject,
                categories: rowsCategories,
                subcategories: rowsSubcategories
            },
        });

    } catch (error) {
        console.error("Error obteniendo configuraciones:", error);
        res.status(500).json({ error: "Error obteniendo configuraciones", details: error.message });
    } finally {
        if (connection) connection.release();
    }
};
export const saveCategory = async (req, res) => {
    let connection;
    const { table_prefix, name } = req.body;

    if (!table_prefix) {
        return res.status(400).json({ error: "Debes enviar table_prefix" });
    }

    if (!name) {
        return res.status(400).json({ error: "Debes enviar el nombre de la categoría" });
    }

    try {
        connection = await getConnection();
        const tableName = `${table_prefix}_categories`;

        // Crear tabla si no existe
        await connection.query(`
            CREATE TABLE IF NOT EXISTS \`${tableName}\` (
                id INT NOT NULL AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL UNIQUE
            )
        `);

        // ¿Ya existe?
        const [rows] = await connection.query(
            `SELECT id FROM \`${tableName}\` WHERE name = ?`,
            [name]
        );

        if (rows.length > 0) {
            return res.status(400).json({ error: "La categoría ya existe" });
        }

        // Insertar
        const [result] = await connection.query(
            `INSERT INTO \`${tableName}\` (name) VALUES (?)`,
            [name]
        );

        res.json({
            msg: "Categoría guardada correctamente",
            data: { id: result.insertId, name }
        });

    } catch (error) {
        console.error("Error guardando categoría:", error);
        res.status(500).json({ error: "Error guardando categoría", details: error.message });
    } finally {
        if (connection) connection.release();
    }
};
export const saveSubcategory = async (req, res) => {
    let connection;
    const { table_prefix, name, category_id } = req.body;

    if (!table_prefix) {
        return res.status(400).json({ error: "Debes enviar table_prefix" });
    }

    if (!name || !category_id) {
        return res.status(400).json({ error: "Debes enviar el nombre y la categoría padre" });
    }

    try {
        connection = await getConnection();
        const categoriesTable = `${table_prefix}_categories`;
        const subcategoriesTable = `${table_prefix}_subcategories`;

        // Crear tablas si no existen
        await connection.query(`
            CREATE TABLE IF NOT EXISTS \`${categoriesTable}\` (
                id INT NOT NULL AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL UNIQUE
            )
        `);

        await connection.query(`
            CREATE TABLE IF NOT EXISTS \`${subcategoriesTable}\` (
                id INT NOT NULL AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(255) NOT NULL,
                category_id INT NOT NULL,
                FOREIGN KEY (category_id) REFERENCES \`${categoriesTable}\`(id) ON DELETE CASCADE
            )
        `);

        // Validar que la categoría exista
        const [cat] = await connection.query(
            `SELECT id FROM \`${categoriesTable}\` WHERE id = ?`,
            [category_id]
        );

        if (cat.length === 0) {
            return res.status(400).json({ error: "La categoría padre no existe" });
        }

        // Validar que no exista la subcategoría en esa categoría
        const [rows] = await connection.query(
            `SELECT id FROM \`${subcategoriesTable}\` WHERE name = ? AND category_id = ?`,
            [name, category_id]
        );

        if (rows.length > 0) {
            return res.status(400).json({ error: "La subcategoría ya existe en esta categoría" });
        }

        // Insertar
        const [result] = await connection.query(
            `INSERT INTO \`${subcategoriesTable}\` (name, category_id) VALUES (?, ?)`,
            [name, category_id]
        );

        res.json({
            msg: "Subcategoría guardada correctamente",
            data: { id: result.insertId, name, category_id }
        });

    } catch (error) {
        console.error("Error guardando subcategoría:", error);
        res.status(500).json({ error: "Error guardando subcategoría", details: error.message });
    } finally {
        if (connection) connection.release();
    }
};
export const getYearsFromReports = async (req, res) => {
    let connection;
    const { table_prefix } = req.body;

    if (!table_prefix) {
        return res.status(400).json({ error: "Debes enviar table_prefix" });
    }

    try {
        connection = await getConnection();

        const [consult] = `SELECT * FROM ${table_prefix}_report_history`
        const result = await connection.query(consult)

        res.json({
            msg: "Consulta realizada correctamente",
            data: { result }
        });

    } catch (error) {
        console.error(error)
    } finally {
        if (connection) connection.release();
    }
}