import instance from "../../../utils/axios.instance";
import { useState } from "react";

/**
 * Hook para enviar reports (regionales o nacionales) al backend
 * @returns { sendReports, loading, error, response }
 */
export default function useReports() {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [response, setResponse] = useState(null);

  /**
   * Envía los reportes al backend
   * @param {Object} params
   * @param {string} params.tablePrefix - Prefijo de la tabla (por ej. 'acoplasticos')
   * @param {number} params.companyId - ID de la empresa
   * @param {string} params.reportType - Tipo de reporte ('region' o 'country')
   * @param {Array} params.reports - Arreglo de reportes
   */
  const sendReports = async ({ tablePrefix, companyId, reportType, reports }) => {
    if (!tablePrefix || !companyId || !Array.isArray(reports) || reports.length === 0) {
      setError("Debe enviar tablePrefix, companyId y un arreglo de reports válidos");
      console.warn("Parámetros inválidos:", { tablePrefix, companyId, reports });
      return;
    }

    if (!["region", "country"].includes(reportType)) {
      setError("Debe especificar reportType como 'region' o 'country'");
      console.warn("reportType inválido:", reportType);
      return;
    }

    setLoading(true);
    setError(null);
    setResponse(null);

    const payload = {
      table_prefix: tablePrefix,
      company_id: companyId,
      report_type: reportType,
      reports,
    };

    try {
      const res = await instance.post("/api/reports", payload);
      setResponse(res.data);
    } catch (err) {
      console.error("Error enviando reports:", err);
      setError(err.response?.data?.error || "Error al enviar los reportes");
    } finally {
      setLoading(false);
    }
  };

  return { sendReports, loading, error, response };
}
