import React, { useState, useEffect, useMemo, useRef } from "react";
import "./home.css";
import {
  PieChart, Pie, Cell, Tooltip, Legend, ResponsiveContainer,
  LineChart, Line, CartesianGrid, XAxis, YAxis
} from "recharts";
import useReports from "../home/hooks/useHome";
import useGetConfigurations from "../configuration/hooks/useGetConfigurations";

import amazonIcon from "../../assets/regions/Amazon-Colombia.webp";
import AndeanIcon from "../../assets/regions/Andean-Colombia.webp";
import CaribbeanIcon from "../../assets/regions/Caribbean-Colombia.webp";
import OrinoquiaIcon from "../../assets/regions/Orinoquia-Colombia.webp";
import PacificIcon from "../../assets/regions/Pacific-Colombia.webp";


//

import closeIcon from "../../assets/icons/xmark-black.svg"
import closeIconSoft from "../../assets/icons/xmark.svg"

export default function Home() {
  const isActive = (value) => value !== "" && value !== null && value !== undefined;

  const [reportesRegion, setReportesRegion] = useState([]);
  const [selectedYear, setSelectedYear] = useState(null);
  const [categoriaSeleccionada, setCategoriaSeleccionada] = useState(null);

  const [selectedMonthRange, setSelectedMonthRange] = useState({ from: "", to: "" });
  const [selectedCategoryId, setSelectedCategoryId] = useState("");
  const [selectedSubcategoryId, setSelectedSubcategoryId] = useState("");
  const [selectedRegion, setSelectedRegion] = useState("");

  const yearRef = useRef(null);
  const monthRef = useRef(null);
  const categoryRef = useRef(null);
  const subcategoryRef = useRef(null);
  const regionRef = useRef(null);

  const limpiarTodo = () => {
    setSelectedYear("");
    setSelectedMonthRange({ from: "", to: "" });
    setSelectedCategoryId("");
    setSelectedSubcategoryId("");
    setSelectedRegion("");
  };

  const limpiarYear = () => setSelectedYear("");
  const limpiarMonth = () => setSelectedMonthRange({ from: "", to: "" });
  const limpiarCategory = () => {
    setSelectedCategoryId("");
    setSelectedSubcategoryId("");
  };
  const limpiarSubcategory = () => setSelectedSubcategoryId("");
  const limpiarRegion = () => setSelectedRegion("");

  const dataFiltrada = useMemo(() => {
    return reportesRegion.filter(r => {
      if (selectedYear && r.year !== selectedYear) return false;
      if (selectedMonthRange.from && selectedMonthRange.to) {
        const m = r.month;
        if (m < Number(selectedMonthRange.from) || m > Number(selectedMonthRange.to)) return false;
      }
      if (selectedCategoryId && r.category_id !== selectedCategoryId) return false;
      if (selectedSubcategoryId && r.subcategory_id !== selectedSubcategoryId) return false;
      if (selectedRegion && r.region !== selectedRegion) return false;
      return true;
    });
  }, [
    reportesRegion,
    selectedYear,
    selectedMonthRange,
    selectedCategoryId,
    selectedSubcategoryId,
    selectedRegion
  ]);



  const { fetchReportsByCompany } = useReports();
  const user = JSON.parse(localStorage.getItem("user"));
  const { table_prefix, company_nit } = user || {};

  const { data: configs } = useGetConfigurations(table_prefix);
  const categories = useMemo(() => configs?.categories || [], [configs]);

  const regiones = [
    { id: 1, nombre: "Caribe", icon: CaribbeanIcon },
    { id: 2, nombre: "Pacífico", icon: PacificIcon },
    { id: 3, nombre: "Andina", icon: AndeanIcon },
    { id: 4, nombre: "Orinoquía", icon: OrinoquiaIcon },
    { id: 5, nombre: "Amazonía", icon: amazonIcon },
  ];

  useEffect(() => {
    const fetchData = async () => {
      try {
        const reportsByCompany = await fetchReportsByCompany({ table_prefix, company_nit });
        setReportesRegion(reportsByCompany.data || []);
      } catch (error) {
        console.error("Error cargando reportes:", error);
      }
    };
    fetchData();
  }, [table_prefix, company_nit]);

  // Obtener años únicos
  const years = useMemo(() => {
    const uniqueYears = Array.from(new Set(dataFiltrada.map(r => r.year)));
    return uniqueYears.sort((a, b) => b - a);
  }, [dataFiltrada]);

  // Pie en porcentaje
  const getPieDataByCategoryPercentage = (categoryId, type) => {
    const filtered = dataFiltrada.filter(
      r => r.category_id === categoryId && (!selectedYear || r.year === selectedYear)
    );
    const total = filtered.reduce((sum, r) => sum + Number(r[type]), 0);
    return regiones.map(region => {
      const value = filtered.filter(r => r.region === region.id)
        .reduce((sum, r) => sum + Number(r[type]), 0);
      const percentage = total > 0 ? (value / total) * 100 : 0;
      return { name: region.nombre, value: percentage };
    });
  };

  const COLORS = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042", "#A28EFF"];

  // Generar tendencia por categoría incluyendo subcategorías
  const generarTendenciaPorCategoria = (categoria) => {
    if (!categoria || !dataFiltrada.length) return [];

    const subcatIds = categoria.subcategorias?.map(sub => sub.id) || [];
    const filtrados = dataFiltrada.filter(
      r => r.category_id === categoria.id || subcatIds.includes(r.subcategory_id)
    );

    const meses = ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"];
    const dataTendencia = [];

    for (let year = 2019; year <= 2025; year++) {
      meses.forEach(mes => {
        const registros = filtrados.filter(r => r.year === year && r.month === Number(mes));
        dataTendencia.push({
          mes: `${mes}/${year}`,
          produccion: registros.reduce((sum, r) => sum + Number(r.produccion || 0), 0),
          ventasGalones: registros.reduce((sum, r) => sum + Number(r.ventas_volumen || 0), 0),
          ventasPesos: registros.reduce((sum, r) => sum + Number(r.ventas_pesos || 0), 0),
          highlight: selectedYear === year
        });
      });
    }
    return dataTendencia;
  };

  const subcategories = useMemo(() => configs?.subcategories || [], [configs]);

  const categoriesWithSubs = useMemo(() => {
    return categories.map(cat => ({
      ...cat,
      subcategories: subcategories.filter(sub => sub.category_id === cat.id)
    }));
  }, [categories, subcategories]);

  return (
    <div className="home-main-container">

      <div className="home-main-header">
        <div>
          Reporte Estudios de Mercado
        </div>
      </div>

      <div className="filtros-container">

        {/* Año */}
        <div onClick={() => yearRef.current?.focus()}>
          <label
            style={{ color: isActive(selectedYear) ? "var(--color-primary)" : "inherit" }}
          >
            <img
              src={closeIcon}
              alt=""
              style={{
                opacity: isActive(selectedYear) ? 0.6 : 0,
                transition: "opacity 0.25s ease"
              }}
              onClick={(e) => { e.stopPropagation(); limpiarYear(); }}
            />
            Año |
          </label>

          <select
            ref={yearRef}
            value={selectedYear || ""}
            onChange={(e) => setSelectedYear(Number(e.target.value))}
          >
            <option value="">Todos</option>
            {years.map(year => <option key={year} value={year}>{year}</option>)}
          </select>
        </div>

        {/* Mes */}
        {/* Mes (rango) */}
        <div>
          <label
            style={{ color: selectedMonthRange.from && selectedMonthRange.to ? "var(--color-primary)" : "inherit" }}
          >
            <img
              src={closeIcon}
              alt=""
              style={{
                opacity: selectedMonthRange.from && selectedMonthRange.to ? 0.6 : 0,
                transition: "opacity 0.25s ease"
              }}
              onClick={(e) => {
                e.stopPropagation();
                setSelectedMonthRange({ from: "", to: "" });
              }}
            />
            Mes |
          </label>

          <div className="month-range">
            <select
              value={selectedMonthRange.from}
              onChange={(e) =>
                setSelectedMonthRange((prev) => ({ ...prev, from: e.target.value }))
              }
            >
              <option value="">Desde</option>
              {[...Array(12)].map((_, i) => (
                <option key={i + 1} value={i + 1}>
                  {new Date(0, i).toLocaleString("es-CO", { month: "long" })}
                </option>
              ))}
            </select>

            <span>-</span>

            <select
              value={selectedMonthRange.to}
              onChange={(e) =>
                setSelectedMonthRange((prev) => ({ ...prev, to: e.target.value }))
              }
            >
              <option value="">Hasta</option>
              {[...Array(12)].map((_, i) => (
                <option key={i + 1} value={i + 1}>
                  {new Date(0, i).toLocaleString("es-CO", { month: "long" })}
                </option>
              ))}
            </select>
          </div>
        </div>


        {/* Categoría */}
        <div onClick={() => categoryRef.current?.focus()}>
          <label
            style={{ color: isActive(selectedCategoryId) ? "var(--color-primary)" : "inherit" }}
          >
            <img
              src={closeIcon}
              alt=""
              style={{
                opacity: isActive(selectedCategoryId) ? 0.6 : 0,
                transition: "opacity 0.25s ease"
              }}
              onClick={(e) => { e.stopPropagation(); limpiarCategory(); }}
            />
            Categoría |
          </label>

          <select
            ref={categoryRef}
            value={selectedCategoryId}
            onChange={(e) => setSelectedCategoryId(Number(e.target.value))}
          >
            <option value="">Todas</option>
            {categories.map(c => <option key={c.id} value={c.id}>{c.name}</option>)}
          </select>
        </div>

        {/* Subcategoría */}
        <div onClick={() => subcategoryRef.current?.focus()}>
          <label
            style={{ color: isActive(selectedSubcategoryId) ? "var(--color-primary)" : "inherit" }}
          >
            <img
              src={closeIcon}
              alt=""
              style={{
                opacity: isActive(selectedSubcategoryId) ? 0.6 : 0,
                transition: "opacity 0.25s ease"
              }}
              onClick={(e) => { e.stopPropagation(); limpiarSubcategory(); }}
            />
            Subcategoría |
          </label>

          <select
            ref={subcategoryRef}
            value={selectedSubcategoryId}
            onChange={(e) => setSelectedSubcategoryId(Number(e.target.value))}
            disabled={!selectedCategoryId}
          >
            <option value="">Todas</option>
            {subcategories
              .filter(sub => sub.category_id === selectedCategoryId)
              .map(sub => (
                <option key={sub.id} value={sub.id}>{sub.name}</option>
              ))}
          </select>
        </div>

        {/* Región */}
        <div onClick={() => regionRef.current?.focus()}>
          <label
            style={{ color: isActive(selectedRegion) ? "var(--color-primary)" : "inherit" }}
          >
            <img
              src={closeIcon}
              alt=""
              style={{
                opacity: isActive(selectedRegion) ? 0.6 : 0,
                transition: "opacity 0.25s ease"
              }}
              onClick={(e) => { e.stopPropagation(); limpiarRegion(); }}
            />
            Región |
          </label>

          <select
            ref={regionRef}
            value={selectedRegion}
            onChange={(e) => setSelectedRegion(Number(e.target.value))}
          >
            <option value="">Todas</option>
            {regiones.map(r => <option key={r.id} value={r.id}>{r.nombre}</option>)}
          </select>
        </div>


        {/* Limpiar todo */}
        <div onClick={limpiarTodo} style={{ cursor: "pointer" }}>
          <label style={{ cursor: "pointer" }}>
            <img src={closeIconSoft} alt="" />
            Limpiar Filtros
          </label>
        </div>


      </div>

      <div className="charts-container">
        {categories.map(category => {
          const dataTendencia = generarTendenciaPorCategoria(category);

          return (
            <>
              <div key={category.id} className="category-charts">
                <h3>{category.name}</h3>

                <div className="flex-pies">
                  <div className="pie-chart-container" style={{ marginLeft: "40px" }}>
                    <h4>Producción (Gal)</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <PieChart>
                        <Pie
                          data={getPieDataByCategoryPercentage(category.id, "produccion")}
                          dataKey="value"
                          nameKey="name"
                          cx="50%"
                          cy="50%"
                          outerRadius={80}
                          label={({ name, value }) => `${name}: ${value.toFixed(1)}%`}
                        >
                          {regiones.map((_, index) => (
                            <Cell key={index} fill={COLORS[index % COLORS.length]} />
                          ))}
                        </Pie>
                        <Tooltip formatter={(value) => `${value.toFixed(1)}%`} />
                        <Legend />
                      </PieChart>
                    </ResponsiveContainer>
                  </div>

                  <div className="pie-chart-container">
                    <h4>Ventas (COP)</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <PieChart>
                        <Pie
                          data={getPieDataByCategoryPercentage(category.id, "ventas_pesos")}
                          dataKey="value"
                          nameKey="name"
                          cx="50%"
                          cy="50%"
                          outerRadius={80}
                          label={({ name, value }) => `${name}: ${value.toFixed(1)}%`}
                        >
                          {regiones.map((_, index) => (
                            <Cell key={index} fill={COLORS[index % COLORS.length]} />
                          ))}
                        </Pie>
                        <Tooltip formatter={(value) => `${value.toFixed(1)}%`} />
                        <Legend />
                      </PieChart>
                    </ResponsiveContainer>
                  </div>
                </div>
              </div>
              <div key={category.id} className="category-charts">
                <h3>{category.name}</h3>
                {/* Sección de tendencias */}
                <div className="trend-section">
                  <h3>Tendencias - {category.name}</h3>

                  <div className="chart-container">
                    <h4>Producción mensual (Gal)</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <LineChart data={dataTendencia} margin={{ top: 20, right: 30, left: 40, bottom: 20 }}>
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="mes" />
                        <YAxis />
                        <Tooltip formatter={(v) => v.toLocaleString("es-CO")} />
                        <Line
                          type="monotone"
                          dataKey="produccion"
                          stroke="#0088FE"
                          strokeWidth={2}
                          dot={(props) => {
                            const { cx, cy, payload } = props;
                            const color = payload.mes.endsWith(`/${selectedYear}`) ? "#FF0000" : "#0088FE";
                            return <circle cx={cx} cy={cy} r={4} fill={color} stroke="#fff" strokeWidth={1} />;
                          }}
                        />
                      </LineChart>
                    </ResponsiveContainer>
                  </div>

                  <div className="chart-container">
                    <h4>Ventas mensuales (Gal)</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <LineChart
                        data={dataTendencia}
                        margin={{ top: 20, right: 30, left: 40, bottom: 20 }}
                      >
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="mes" />
                        <YAxis />
                        <Tooltip formatter={(v) => v.toLocaleString("es-CO")} />
                        <Line
                          type="monotone"
                          dataKey="ventasGalones"
                          stroke="#00C49F"
                          strokeWidth={2}
                        />
                      </LineChart>

                    </ResponsiveContainer>
                  </div>

                  <div className="chart-container">
                    <h4>Ventas mensuales (COP)</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <LineChart data={dataTendencia} margin={{ top: 20, right: 30, left: 50, bottom: 20 }}>
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="mes" />
                        <YAxis />
                        <Tooltip formatter={(v) => `$${v.toLocaleString("es-CO")}`} />
                        <Line
                          type="monotone"
                          dataKey="ventasPesos"
                          stroke="#FF8042"
                          strokeWidth={2}
                        />
                      </LineChart>
                    </ResponsiveContainer>
                  </div>
                </div>
              </div>
            </>

          );
        })}
        {/* Tabla resumen */}
        <div className="tabla-categorias-container">
          <h3>Resumen por categoría y subcategoría</h3>
          <table className="tabla-categorias">
            <thead>
              <tr>
                <th rowSpan="3">Mes/Año</th>
                {categoriesWithSubs.map(cat => (
                  <th key={`cat-${cat.id}`} colSpan={cat.subcategories.length * 3}>
                    {cat.name}
                  </th>
                ))}
              </tr>
              <tr>
                {categoriesWithSubs.map(cat =>
                  cat.subcategories.map(sub => (
                    <th key={`sub-${sub.id}`} colSpan="3">{sub.name}</th>
                  ))
                )}
              </tr>
              <tr>
                {categoriesWithSubs.map(cat =>
                  cat.subcategories.map(sub => (
                    <React.Fragment key={`sub-${sub.id}-headers`}>
                      <th>Ventas (COP)</th>
                      <th>Producción (Gal)</th>
                      <th>Ventas (Gal)</th>
                    </React.Fragment>
                  ))
                )}
              </tr>
            </thead>

            <tbody>
              {(() => {
                const dataFiltradaTabla = selectedYear
                  ? dataFiltrada.filter(r => r.year === selectedYear)
                  : dataFiltrada;

                const fechas = Array.from(new Set(dataFiltradaTabla.map(r => `${r.year}-${r.month}`)));

                return fechas.map((ym) => {
                  const [year, month] = ym.split("-");
                  const labelMes = new Date(year, month - 1).toLocaleString("es-CO", { month: "long" });

                  return (
                    <tr key={`row-${ym}`}>
                      <td>{`${labelMes} - ${year}`}</td>

                      {categoriesWithSubs.map(cat =>
                        cat.subcategories.map(sub => {
                          const registros = dataFiltradaTabla.filter(
                            r =>
                              r.year === Number(year) &&
                              r.month === Number(month) &&
                              r.subcategory_id === sub.id
                          );

                          const totalVentas = registros.reduce((s, r) => s + Number(r.ventas_pesos || 0), 0);
                          const totalProduccion = registros.reduce((s, r) => s + Number(r.produccion || 0), 0);
                          const totalGalones = registros.reduce((s, r) => s + Number(r.ventas_volumen || 0), 0);

                          return (
                            <React.Fragment key={`sub-${sub.id}-ym-${year}-${month}`}>
                              <td>{totalVentas ? totalVentas.toLocaleString("es-CO") : "-"}</td>
                              <td>{totalProduccion ? totalProduccion.toLocaleString("es-CO") : "-"}</td>
                              <td>{totalGalones ? totalGalones.toLocaleString("es-CO") : "-"}</td>
                            </React.Fragment>
                          );
                        })
                      )}
                    </tr>
                  );
                });
              })()}
            </tbody>

          </table>
        </div>
      </div>
    </div>
  );
}
