import { PieChart, Pie, Cell, Tooltip, Legend, ResponsiveContainer } from "recharts";
import { motion } from "framer-motion";

const COLORS = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042", "#AF19FF"];

const RegionalParticipation = ({ reportesRegion, categorias }) => {
  // Construir data agrupada por categoría y tipo de valor (ventas, producción)
  const dataPorCategoria = categorias.map((cat) => {
    const dataCategoria = reportesRegion.filter((r) => r.category_id === cat.id && r.report_type === "region");

    // Agrupar por región
    const regiones = {};
    dataCategoria.forEach((item) => {
      const region = item.region ?? "Sin región";
      if (!regiones[region]) {
        regiones[region] = { ventas_pesos: 0, produccion: 0 };
      }
      regiones[region].ventas_pesos += parseFloat(item.ventas_pesos || 0);
      regiones[region].produccion += parseFloat(item.produccion || 0);
    });

    const dataVentas = Object.keys(regiones).map((region) => ({
      name: `Región ${region}`,
      value: regiones[region].ventas_pesos,
    }));

    const dataProduccion = Object.keys(regiones).map((region) => ({
      name: `Región ${region}`,
      value: regiones[region].produccion,
    }));

    return { categoria: cat.nombre, dataVentas, dataProduccion };
  });

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
      {dataPorCategoria.map(({ categoria, dataVentas, dataProduccion }) => (
        <motion.div
          key={categoria}
          className="bg-white shadow-md rounded-2xl p-4"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.4 }}
        >
          <h3 className="text-lg font-semibold mb-4 text-center">{categoria}</h3>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Pie de Ventas */}
            <div>
              <h4 className="text-center mb-2">Participación regional (Ventas $)</h4>
              <ResponsiveContainer width="100%" height={300}>
                <PieChart>
                  <Pie
                    data={dataVentas}
                    cx="50%"
                    cy="50%"
                    outerRadius={100}
                    dataKey="value"
                    label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(1)}%`}
                  >
                    {dataVentas.map((entry, index) => (
                      <Cell key={`ventas-${index}`} fill={COLORS[index % COLORS.length]} />
                    ))}
                  </Pie>
                  <Tooltip formatter={(v) => `$${v.toLocaleString("es-CO")}`} />
                  <Legend />
                </PieChart>
              </ResponsiveContainer>
            </div>

            {/* Pie de Producción */}
            <div>
              <h4 className="text-center mb-2">Participación regional (Producción)</h4>
              <ResponsiveContainer width="100%" height={300}>
                <PieChart>
                  <Pie
                    data={dataProduccion}
                    cx="50%"
                    cy="50%"
                    outerRadius={100}
                    dataKey="value"
                    label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(1)}%`}
                  >
                    {dataProduccion.map((entry, index) => (
                      <Cell key={`prod-${index}`} fill={COLORS[index % COLORS.length]} />
                    ))}
                  </Pie>
                  <Tooltip formatter={(v) => `${v.toLocaleString("es-CO")} unidades`} />
                  <Legend />
                </PieChart>
              </ResponsiveContainer>
            </div>
          </div>
        </motion.div>
      ))}
    </div>
  );
};

export default RegionalParticipation;
