import { useState, useEffect } from "react";
import "./configuration.css";

import useSaveColors from "./hooks/useSaveColors";
import useGetConfigurations from "./hooks/useGetConfigurations";
import useSaveCategory from "./hooks/useCategories";
import useSaveSubcategory from "./hooks/useSubCategories";

export default function Configuration() {
  const [activeTab, setActiveTab] = useState("paleta");

  const loginResponse = localStorage.getItem("loginResponse");
  const user = loginResponse ? JSON.parse(loginResponse).user : null;

  // obtenemos configs desde la DB
  const { data: configs, loading, error } = useGetConfigurations(user?.table_prefix);

  // log cada vez que cambien las configuraciones
  useEffect(() => {
    if (configs) {
      console.log("Configs cargadas desde la DB:", configs);
    }
  }, [configs]);

  // estados locales para los colores (usados en los pickers)
  const [colorPrimary, setColorPrimary] = useState("#000000");
  const [colorSecondary, setColorSecondary] = useState("#000000");
  const [colorTertiary, setColorTertiary] = useState("#000000");
  const [colorQuaternary, setColorQuaternary] = useState("#000000");


  useEffect(() => {
    const storedColors = JSON.parse(localStorage.getItem("researchColors"));
    if (storedColors) {
      setColorPrimary(storedColors.color_primary);
      setColorSecondary(storedColors.color_secondary);
      setColorTertiary(storedColors.color_tertiary);
      setColorQuaternary(storedColors.color_quaternary);
    }

    if (configs?.configurations) {
      const c = configs.configurations;

      const updatedColors = {
        color_primary: c.color_primary || "#000000",
        color_secondary: c.color_secondary || "#000000",
        color_tertiary: c.color_tertiary || "#000000",
        color_quaternary: c.color_quaternary || "#000000",
      };

      setColorPrimary(updatedColors.color_primary);
      setColorSecondary(updatedColors.color_secondary);
      setColorTertiary(updatedColors.color_tertiary);
      setColorQuaternary(updatedColors.color_quaternary);

      localStorage.setItem("researchColors", JSON.stringify(updatedColors));
    }
  }, [configs]);



  // HOOKS colores
  const { saveColors, loading: savingColors } = useSaveColors();

  const handleSaveColors = () => {
    if (!user) return;
    saveColors({
      table_prefix: user.table_prefix,
      color_primary: colorPrimary,
      color_secondary: colorSecondary,
      color_tertiary: colorTertiary,
      color_quaternary: colorQuaternary,
    });
  };

  // HOOKS categorías
  const [categoryName, setCategoryName] = useState("");
  const { saveCategory, loading: savingCategory, error: errorCategory } = useSaveCategory();

  const handleSaveCategory = () => {
    if (!user || !categoryName) return;
    saveCategory({
      table_prefix: user.table_prefix,
      name: categoryName,
    }).then(() => setCategoryName("")); // limpiar input al guardar
  };

  // HOOKS subcategorías
  const [subcategoryName, setSubcategoryName] = useState("");
  const [categoryId, setCategoryId] = useState("");
  const { saveSubcategory, loading: savingSub, error: errorSub } = useSaveSubcategory();

  const handleSaveSubcategory = () => {
    if (!user || !subcategoryName || !categoryId) return;
    saveSubcategory({
      table_prefix: user.table_prefix,
      name: subcategoryName,
      category_id: categoryId,
    }).then(() => {
      setSubcategoryName("");
      setCategoryId("");
    });
  };

  const menuItems = [
    { key: "paleta", label: "Paleta de colores" },
    { key: "categorias", label: "Categorías" },
    { key: "subcategorias", label: "Subcategorías" },
  ];

  const renderContent = () => {
    switch (activeTab) {
      case "paleta":
        if (loading) return <p>Cargando configuraciones...</p>;
        if (error) return <p>Error: {JSON.stringify(error)}</p>;
        return (
          <div className="palette-config">
            <h3 className="section-title">Paleta de Colores</h3>

            <div className="color-picker">
              <label>
                Color Primario
                <input
                  type="color"
                  value={colorPrimary}
                  onChange={(e) => setColorPrimary(e.target.value)}
                />
              </label>
            </div>
            <div className="color-picker">
              <label>
                Color Secundario
                <input
                  type="color"
                  value={colorSecondary}
                  onChange={(e) => setColorSecondary(e.target.value)}
                />
              </label>
            </div>
            <div className="color-picker">
              <label>
                Color Terciario
                <input
                  type="color"
                  value={colorTertiary}
                  onChange={(e) => setColorTertiary(e.target.value)}
                />
              </label>
            </div>
            <div className="color-picker">
              <label>
                Color Cuaternario
                <input
                  type="color"
                  value={colorQuaternary}
                  onChange={(e) => setColorQuaternary(e.target.value)}
                />
              </label>
            </div>
            <button onClick={handleSaveColors} disabled={savingColors}>
              {savingColors ? "Guardando..." : "Guardar colores"}
            </button>
          </div>
        );
      case "categorias":
        return (
          <div className="categories-config">
            <h3 className="section-title">Agregar Categoría</h3>

            <div className="category-form">
              <input
                type="text"
                placeholder="Nombre de categoría"
                value={categoryName}
                onChange={(e) => setCategoryName(e.target.value)}
              />

              <button onClick={handleSaveCategory} disabled={savingCategory}>
                {savingCategory ? "Guardando..." : "Guardar categoría"}
              </button>

              {errorCategory && (
                <p className="error-msg">Error: {JSON.stringify(errorCategory)}</p>
              )}
            </div>

            <h4 className="section-subtitle">Categorías existentes</h4>

            <div className="category-list">
              {configs?.categories?.length > 0 ? (
                configs.categories.map((cat) => (
                  <div className="category-item" key={cat.id}>
                    <span className="category-name">{cat.name}</span>

                    <ul className="subcategory-list">
                      {configs?.subcategories
                        ?.filter((sub) => sub.category_id === cat.id)
                        .map((sub) => (
                          <li key={sub.id}>{sub.name}</li>
                        ))}
                    </ul>
                  </div>
                ))
              ) : (
                <p className="empty-text">No hay categorías registradas</p>
              )}
            </div>
          </div>

        );

      case "subcategorias":
        return (
          <div className="subcategories-config">
            <h3 className="section-title">Agregar Subcategoría</h3>

            <div className="subcategory-form">
              <input
                type="text"
                placeholder="Nombre de subcategoría"
                value={subcategoryName}
                onChange={(e) => setSubcategoryName(e.target.value)}
              />

              <select
                value={categoryId}
                onChange={(e) => setCategoryId(e.target.value)}
              >
                <option value="">Selecciona una categoría</option>
                {configs?.categories?.map((cat) => (
                  <option key={cat.id} value={cat.id}>
                    {cat.name}
                  </option>
                ))}
              </select>

              <button onClick={handleSaveSubcategory} disabled={savingSub}>
                {savingSub ? "Guardando..." : "Guardar"}
              </button>

              {errorSub && (
                <p className="error-msg">
                  Error: {JSON.stringify(errorSub)}
                </p>
              )}
            </div>

            <h4 className="section-subtitle">Categorías y Subcategorías</h4>

            <div className="subcategory-list-container">
              {configs?.categories?.length > 0 ? (
                configs.categories.map((cat) => (
                  <div className="subcategory-item" key={cat.id}>
                    <span className="subcategory-cat-title">{cat.name}</span>

                    <ul className="subcategory-list">
                      {configs?.subcategories
                        ?.filter((sub) => sub.category_id === cat.id)
                        .map((sub) => (
                          <li key={sub.id}>{sub.name}</li>
                        ))}
                    </ul>
                  </div>
                ))
              ) : (
                <p className="empty-text">No hay categorías registradas</p>
              )}
            </div>
          </div>

        );

      default:
        return <div>Selecciona una opción</div>;
    }
  };

  return (
    <div className="configuration-container">
      <div className="config-content">
        <div className="config-content-left">
          {menuItems.map((item) => (
            <div
              key={item.key}
              className={`menu-item ${activeTab === item.key ? "active" : ""}`}
              onClick={() => setActiveTab(item.key)}
            >
              {item.label}
            </div>
          ))}
        </div>
        <div className="config-content-rigth">{renderContent()}</div>
      </div>
    </div>
  );
}
