import "./administration.css";
import { useEffect, useState } from "react";
import useEditableReports from "./hooks/useEditableReports";
import useDeleteReportHistory from "./hooks/useDeleteReportHistory";
import Modal from "../../ui/modal/modal";

export default function Administration() {
  // Obtener usuario y datos del localStorage
  const loginResponse = localStorage.getItem("loginResponse");
  const user = loginResponse ? JSON.parse(loginResponse).user : null;
  const tablePrefix = user?.table_prefix;
  const companyId = user?.company_nit;

  const { fetchEditableReports, data, loading, error } = useEditableReports();
  const { deleteReportHistory, loading: deleting } = useDeleteReportHistory();

  // Estado del modal y selección
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedCompany, setSelectedCompany] = useState(null);
  const [selectedDate, setSelectedDate] = useState({ year: null, month: null });

  // Datos falsos de empresas
  const testCompanies = [
    { id: 1, name: "Test 1", nit: "900111111" },
    { id: 2, name: "Test 2", nit: "900222222" },
    { id: 3, name: "Test 3", nit: "900333333" },
    { id: 4, name: "Test 4", nit: "900444444" },
    { id: 5, name: "Test 5", nit: "900555555" },
  ];

  // Datos falsos de historial de reportes
  const historyData = {
    history: [
      { year: 2024, month: 3 },
      { year: 2024, month: 5 },
      { year: 2025, month: 1 },
      { year: 2025, month: 8 },
    ],
  };

  // Logs de eventos inventados
  const eventLogs = [
    { id: 1, company: "Test 1", date: "2025-10-27", action: "Ha reportado el mes de septiembre 2025." },
    { id: 2, company: "Test 2", date: "2025-10-26", action: "Ha solicitado permiso de edición para agosto 2025." },
    { id: 3, company: "Test 3", date: "2025-10-25", action: "Se le ha rechazado la solicitud de edición del mes de julio 2025." },
    { id: 4, company: "Test 4", date: "2025-10-23", action: "Ha reportado correctamente el mes de agosto 2025." },
    { id: 5, company: "Test 5", date: "2025-10-20", action: "Ha solicitado permiso de edición para junio 2025." },
    { id: 6, company: "Test 1", date: "2025-09-29", action: "Se ha aprobado la solicitud de edición del mes de mayo 2025." },
    { id: 7, company: "Test 3", date: "2025-09-20", action: "Ha reportado el mes de abril 2025 con observaciones." },
    { id: 8, company: "Test 4", date: "2025-09-15", action: "Ha solicitado permiso de edición para marzo 2025." },
    { id: 9, company: "Test 2", date: "2025-08-30", action: "Ha completado el envío del reporte de julio 2025." },
    { id: 10, company: "Test 5", date: "2025-08-25", action: "Ha sido rechazado el reporte de mayo 2025 por inconsistencias." },
    { id: 11, company: "Test 1", date: "2025-08-15", action: "Ha reportado exitosamente el mes de abril 2025." },
    { id: 12, company: "Test 2", date: "2025-08-10", action: "Ha solicitado corrección del reporte de marzo 2025." },
    { id: 13, company: "Test 3", date: "2025-07-29", action: "Ha enviado reporte pendiente de febrero 2025." },
    { id: 14, company: "Test 4", date: "2025-07-10", action: "Ha sido aprobada la edición de enero 2025." },
    { id: 15, company: "Test 5", date: "2025-06-15", action: "Ha reportado el mes de diciembre 2024 correctamente." },
  ];

  useEffect(() => {
    if (tablePrefix && companyId) {
      fetchEditableReports({
        table_prefix: tablePrefix,
        company_nit: companyId,
      });
    }
  }, [tablePrefix, companyId]);

  const handleGiveEditPermission = async (year, month) => {
    const confirmDelete = window.confirm(
      `¿Dar permiso de edición para el reporte de ${month}/${year}?`
    );
    if (!confirmDelete) return;

    try {
      await deleteReportHistory({
        table_prefix: tablePrefix,
        company_nit: companyId,
        year,
        month,
      });

      await fetchEditableReports({
        table_prefix: tablePrefix,
        company_nit: companyId,
      });
    } catch (err) {
      console.error("Error al eliminar el registro:", err);
    }
  };

  const handleOpenCompanyModal = (company) => {
    setSelectedCompany(company);
    setIsModalOpen(true);
  };

  const handleOpenForm = () => {
    alert(
      `Continuar con ${selectedDate.month}/${selectedDate.year} para ${selectedCompany?.name}`
    );
    setIsModalOpen(false);
  };

  return (
    <div className="administration-container">
      <h2>Administración de reportes</h2>

      {loading && <p>Cargando reportes editables...</p>}
      {error && <p style={{ color: "red" }}>Error: {error.error}</p>}

      {data?.reports?.length > 0 ? (
        <table className="editable-reports-table">
          <thead>
            <tr>
              <th>ID</th>
              <th>Año</th>
              <th>Mes</th>
              <th>NIT Empresa</th>
              <th>Acción</th>
            </tr>
          </thead>
          <tbody>
            {data.reports.map((r) => (
              <tr key={r.id}>
                <td>{r.id}</td>
                <td>{r.year}</td>
                <td>{r.month}</td>
                <td>{r.company_nit}</td>
                <td>
                  <button
                    className="grant-edit-btn"
                    disabled={deleting}
                    onClick={() => handleGiveEditPermission(r.year, r.month)}
                  >
                    Dar permiso de edición
                  </button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      ) : (
        !loading && <p>No hay reportes con permiso de edición.</p>
      )}

      {/* --- SECCIÓN DE EMPRESAS DE PRUEBA --- */}
      <h3 style={{ marginTop: "2rem" }}>Empresas</h3>
      <table className="editable-reports-table">
        <thead>
          <tr>
            <th>ID</th>
            <th>Empresa</th>
            <th>NIT</th>
            <th>Acción</th>
          </tr>
        </thead>
        <tbody>
          {testCompanies.map((c) => (
            <tr key={c.id}>
              <td>{c.id}</td>
              <td>{c.name}</td>
              <td>{c.nit}</td>
              <td>
                <button onClick={() => handleOpenCompanyModal(c)}>
                  Ver meses faltantes
                </button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>

      {/* --- MODAL --- */}
      <Modal isOpen={isModalOpen}>
        <div className="report-date-selection">
          <div className="report-date-title">
            Seleccionar mes y año del reporte - {selectedCompany?.name}
          </div>

          <div className="year-buttons-container">
            {[2019, 2020, 2021, 2022, 2023, 2024, 2025].map((year) => (
              <div key={year} className="year-block">
                <h3>{year}</h3>
                <div className="months-grid">
                  {[
                    "Enero",
                    "Febrero",
                    "Marzo",
                    "Abril",
                    "Mayo",
                    "Junio",
                    "Julio",
                    "Agosto",
                    "Septiembre",
                    "Octubre",
                    "Noviembre",
                    "Diciembre",
                  ].map((mes, i) => {
                    const monthNum = i + 1;
                    const isReported = historyData.history.some(
                      (r) => r.year === year && r.month === monthNum
                    );

                    return (
                      <button
                        key={`${year}-${monthNum}`}
                        className={`month-button 
                          ${selectedDate.year === year && selectedDate.month === monthNum ? "selected" : ""}
                          ${isReported ? "reported" : "missing"}
                        `}
                        onClick={() =>
                          setSelectedDate({ year, month: monthNum })
                        }
                      >
                        {mes}
                      </button>
                    );
                  })}
                </div>
              </div>
            ))}
          </div>

          <div className="btn-container">
            <button
              className="continue-button"
              onClick={handleOpenForm}
              disabled={!selectedDate.year || !selectedDate.month}
            >
              Continuar
            </button>

            <button
              className="close-modal"
              onClick={() => setIsModalOpen(false)}
            >
              Cerrar
            </button>
          </div>


        </div>
      </Modal>

      {/* --- LOG DE EVENTOS --- */}
      <h3 style={{ marginTop: "2rem" }}>Log de eventos y acciones</h3>
      <table className="event-log-table">
        <thead>
          <tr>
            <th>#</th>
            <th>Empresa</th>
            <th>Fecha</th>
            <th>Acción / Evento</th>
          </tr>
        </thead>
        <tbody>
          {eventLogs.map((log) => (
            <tr key={log.id}>
              <td>{log.id}</td>
              <td>{log.company}</td>
              <td>{log.date}</td>
              <td>{log.action}</td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
}
